/*
** Source Lines of Code Counter      (c) Copyright IBM Corp., 1987,1992
**                                   For Internal Use Only
**    by: Jeffrey W. Hamilton  (JEFFH at WMAVM7)
*/

#include "global.h"

extern char *profile_fn;

/*
** To reduce the complexity of the code, I've switched the counting routine
** to a state machine. The following define the various states.
*/
#define START_STATE  0
#define IN_COMMENT   1
#define IN_IGNORE    2
#define IN_ESCAPE    3
#define IN_LINE      4
#define ALTER_STATE  5

int start;     /* Start of physical line */
int end;       /* End of physical line */
int at_start;  /* true when we are at the beginning of a logical line */
int at_line_start; /* true when we are at the beginning of a logical line */
int column;    /* Current offset in physical line */
int field;     /* Current field in physcial line */
struct {
   int type;                  /* Type of state */
   int nesting_level;         /* Depth of nesting so far */
   sym_marker *nest_marker;   /* This state is nestable */
   sym_marker *end_marker;    /* Must match to exit this state */
   } state[500];
int index;                    /* Current state */

/*
** Search string from beginning to end to see if it matches the
** given marker.
*/
int unanchored_match(char *string,sym_marker *marker)
{
   int i,offset;

   switch (marker->type) {
   case AT_START:
   case AT_LINE_START:
      return re_search(marker->expression, string);
   case FIELD:
      /* Advance to the proper field */
      i = 0;
      for (offset = 1; offset < marker->position; offset++) {
         /* Skip non-blank characters */
         while (!isspace(string[i])) {
            /* Check for insufficent fields */
            if (string[i] == '\0') return -1;
            i += 1;
         }
         /* Skip blank characters */
         while (isspace(string[i])) i += 1;
         if (string[i] == '\0') return -1;
      }
      return re_search(marker->expression, &string[i]);
   case COLUMN:
      if (strlen(string) < marker->position) {
         return -1;
      } else {
         return re_search(marker->expression, &string[marker->position - 1]) + marker->position - 1;
      }
   default:
      return re_search_unanchored(marker->expression, string, &offset);
   }
}

/*
** Do an anchored match against a given expression.
*/
int anchored_match(char *string, sym_marker *marker, int *length)
{

   if (marker == NULL) return 0;

   switch (marker->type) {
   case AT_START:
      if (!at_start) return 0;
      break;
   case AT_LINE_START:
      if (!at_line_start) return 0;
      break;
   case FIELD:
      if (field != marker->position) return 0;
      break;
   case COLUMN:
      if (column != marker->position) return 0;
      break;
   }

   if ((*length = re_search(marker->expression, string)) == -1) {
      /* Not found */
      return 0;
   } else {
      /* Found */
      return 1;
   }
}

/*
** Do an anchored search on all pairs in the given set. If starting
** expression is found, return the address of the pair.
*/
static sym_pair *search_match(char *string, sym_pair *range,
                              int *length)
{
   while (range != NULL) {
      if (anchored_match(string, &range->start, length)) {
         return range;
      }
      range = range->next;
   }
   return NULL;
}

/*
** Do an anchored search on all modes in the given description. If starting
** expression is found, return the address of the pair and the address of the
** new mode
*/
static sym_pair *search_mode(char *string, alt_mode *mode,
                             alt_mode **new_mode, int *length)
{
   sym_pair *result;

   while (mode != NULL) {
      if ((result = search_match(string, mode->alternate, length)) != NULL) {
         *new_mode = mode->next;
         return result;
      }
      mode = mode->next;
   }
   return NULL;
}

/*
** Perform a count for a file
*/
void record_count(sym_list *item)
{
   FILE *fp;
   char drive[DISKNAME_LIMIT+1];
   char dir[PATHNAME_LIMIT+1];
   char filename[FILENAME_LIMIT+1];
   char extension[EXTENSION_LIMIT+2];
   description *temp_descr;
   alt_mode *mode;
   char string[256];
   int count_code;
   int cont_on_next;
   sym_pair *result;
   register int i;
   int length, last_length;

   /*
   ** Open the source file
   */
   if ((fp = fopen(item->name,"r")) == NULL) {
      fprintf(stderr,"Unable to open file %s for reading\n",item->name);
      exit(4);
   }

   /*
   ** Locate file extension in description list
   */
   fnsplit(item->name, drive, dir, filename, extension);
#if defined(OS_DOS) || defined(OS_OS2) || defined(OS_VM) || defined(OS_MVS)
   /* These systems are generally case insensitive for file names */
   /* Convert the extension to the same case as the profile */
   strlwr(extension);
#endif
   for (temp_descr = descr; temp_descr != NULL; temp_descr = temp_descr->next) {
      /* Start check by skipping over the leading '.' */
      if (re_search(temp_descr->ext,&extension[1]) != -1) break;
   }
   if (temp_descr == NULL) {
      fprintf(stderr,"No language description for extension %s in %s\n",extension,profile_fn);
   } else {
      mode = temp_descr->mode;
      total_files += 1;
      cont_on_next = 0;
      count_code = 0;
      state[0].type = START_STATE;
      state[0].end_marker = NULL;
      index = 0;
      while (fgets(string,sizeof(string),fp) != NULL) {
         /* Start of another line */
         item->lines++;
         if (debug) printf("Line %d:\n%s\n",item->lines,string);

         last_length = 0;

         /* Determine if this line is a continuation */
         if (cont_on_next) {
            /* The previous line had a continuation marker */
            at_start = 0;
         } else if ((mode->cont_prev != NULL) &&
                    (unanchored_match(string, &mode->cont_prev->start) != -1)) {
            /* This line has a continuation marker */
            if (debug) printf("Continue from previous line detected\n");
            at_start = 0;
         } else {
            at_start = 1;
         }

         /* Check if the next line is a continuation */
         if ((mode->cont_next != NULL) &&
             (unanchored_match(string, &mode->cont_next->start) != -1)) {
            /* Next line will be a continuation */
            if (debug) printf("Continue on next line detected\n");
            cont_on_next = 1;
         } else {
            cont_on_next = 0;
         }

         /* Determine how much of the line we can see */
         if (mode->view != NULL) {
            /* Restricted view of line */
            start = unanchored_match(string,&mode->view->start);
            if (start == -1) {
               start = 0;
            }
            end = unanchored_match(string,&mode->view->end);
            if (end == -1) {
               end = strlen(string)-1;
            } else {
               /* Force termination of the string */
               string[end+1] = '\0';
            }
            if (debug) printf("Restricted View from column %d to column %d\n",
			      start,end);
         } else {
            /* No restriction on the view */
            start = 0;
            end = strlen(string)-1;
         }

         field = 1;
         /* Scan the line, a character at at time */
         i = start;
         while ((i <= end) && (string[i] != '\0')) {
            column = i+1;
            switch (state[index].type) {
            case START_STATE:
               /*
               ** We are not looking for the end of anything in this state
               ** so we can skip the last ending marker in the line.
               */
               if (last_length != 0) {
                  i += last_length;
                  last_length = 0;
               } else if ((result = search_match(&string[i], mode->comment, &length)) != NULL) {
                  item->comments += 1;
                  if (debug) printf("Start comment line %d in column %d\n",item->comments,i);
                  i += length;
                  index++;
                  state[index].type = IN_COMMENT;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
               } else if ((result = search_match(&string[i], mode->line, &length)) != NULL) {
                  if (debug) printf("Start line %d in column %d\n",
                                     item->code+1, i);
                  at_line_start = 1;
                  index++;
                  state[index].type = IN_LINE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  /*
                  ** For lines, we don't advance the line offset so we can
                  ** check for escapes, ignores, and alternative modes.
                  */
               } else {
                  /* No match, advance to the next character */
                  i += 1;
               }
               break;
            case IN_COMMENT:
               if (at_start) {
                  item->comments++;
                  if (debug) printf("Start comment line %d in column %d\n",item->comments,i);
               }
               if (anchored_match(&string[i], state[index].end_marker, &length)) {
                  if (state[index].nesting_level != 0) {
                     if (debug) printf("End comment nested %d levels in column %d\n",
                                        state[index].nesting_level, i);
                     state[index].nesting_level--;
                     i += length;
                  } else {
                     if (debug) printf("End comment in column %d\n",i);
                     index--;
                     /*
                     ** For ending comments, we don't advance the line offset
                     ** so we can check for end-lines, end-ignores, and
                     ** end-alternative modes.
                     */
                     last_length = length;
                  }
               } else if (last_length != 0) {
                  /*
                  ** Before checking for starting markers, we need to skip
                  ** any outstanding ending markers that we delayed.
                  */
                  i += last_length;
                  last_length = 0;
               } else if (anchored_match(&string[i], state[index].nest_marker, &length)) {
                  state[index].nesting_level++;
                  if (debug) printf("Start comment nested %d levels in column %d\n",
                                     state[index].nesting_level,i);
                  i += length;
               } else {
                  /* Nothing matches, continue with the next character */
                  i += 1;
               }
               break;
            case IN_IGNORE:
               if (anchored_match(&string[i], state[index].end_marker, &length)) {
                  if (state[index].nesting_level != 0) {
                     if (debug) printf("End ignore nested %d levels in column %d\n",
                                        state[index].nesting_level, i);
                     state[index].nesting_level--;
                     i += length;
                  } else {
                     if (debug) printf("End ignore in column %d\n",i);
                     index--;
                     /*
                     ** For ending ingnore, we don't advance the line offset
                     ** so we can check for end-lines, end-comments, and
                     ** end-alternative modes.
                     */
                     last_length = length;
                  }
               } else if (last_length != 0) {
                  /*
                  ** Before checking for starting markers, we need to skip
                  ** any outstanding ending markers that we delayed.
                  */
                  i += last_length;
                  last_length = 0;
               } else if (anchored_match(&string[i], state[index].nest_marker, &length)) {
                  state[index].nesting_level++;
                  if (debug) printf("Start ignore nested %d levels in column %d\n",
                                     state[index].nesting_level,i);
                  i += length;
               } else if ((result = search_match(&string[i], mode->escape, &length)) != NULL) {
                  count_code = 1;
                  if (debug) printf("Start escape in column %d\n",i);
                  index++;
                  state[index].type = IN_ESCAPE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else {
                  /* Nothing matches, continue to the next character */
                  i += 1;
               }
               break;
            case IN_ESCAPE:
               if (anchored_match(&string[i], state[index].end_marker, &length)) {
                  if (debug) printf("End escape in column %d\n",i);
                  index--;
                  i += length;
               } else {
                  /* Nothing matches, continue to the next character */
                  i += 1;
               }
               break;
            case IN_LINE:
               if (anchored_match(&string[i], state[index].end_marker, &length)) {
                  /* Count only non-blank lines */
                  if (count_code) {
                     item->code += 1;
                     count_code = 0;
                  } else {
                     if (debug) printf("Source line empty - not counted\n");
                  }
                  if (debug) printf("End line %d in column %d\n",
                                     item->code,i);
                  index--;
                  last_length = length;
               } else if (last_length != 0) {
                  /*
                  ** Before checking for starting markers, we need to skip
                  ** any outstanding ending markers that we delayed.
                  */
                  i += last_length;
                  last_length = 0;
               } else if ((result = search_match(&string[i], mode->ignore, &length)) != NULL) {
                  count_code = 1;
                  if (debug) printf("Start ignore in column %d\n",i);
                  index++;
                  state[index].type = IN_IGNORE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else if ((result = search_match(&string[i], mode->comment, &length)) != NULL) {
                  /* Lines with only comments are empty source lines */
                  item->comments += 1;
                  if (debug) printf("Start comment line %d in column %d\n",item->comments,i);
                  index++;
                  state[index].type = IN_COMMENT;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else if ((result = search_match(&string[i], mode->escape, &length)) != NULL) {
                  count_code = 1;
                  if (debug) printf("Start escape in column %d\n",i);
                  index++;
                  state[index].type = IN_ESCAPE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else if ((result = search_mode(&string[i], temp_descr->mode, &mode, &length)) != NULL) {
                  count_code = 1;
                  if (debug) printf("Start alternative in column %d\n",i);
                  index++;
                  state[index].type = ALTER_STATE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else {
                  /* Nothing Matches */
                  if (!isspace(string[i])) {
                     /* This line actually contains something! */
                     count_code = 1;
                  }
                  i += 1;
               }
               at_line_start = 0;
               break;
            case ALTER_STATE:
               if (anchored_match(&string[i], state[index].end_marker, &length)) {
                  if (state[index].nesting_level != 0) {
                     if (debug) printf("End alternative nested %d levels in column %d\n",
                                        state[index].nesting_level, i);
                     state[index].nesting_level--;
                     i += length;
                  } else {
                     if (debug) printf("End alternative in column %d\n",i);
                     /* Switch back to the main mode of counting */
                     mode = temp_descr->mode;
                     index--;
                     /*
                     ** For ending alternative, we don't advance the line offset
                     ** so we can check for end-lines, end-comments, and
                     ** end-ignores.
                     */
                     last_length = length;
                  }
               } else if (last_length != 0) {
                  /*
                  ** Before checking for starting markers, we need to skip
                  ** any outstanding ending markers that we delayed.
                  */
                  i += last_length;
                  last_length = 0;
               } else if (anchored_match(&string[i], state[index].nest_marker, &length)) {
                  state[index].nesting_level++;
                  if (debug) printf("Start alternative nested %d levels in column %d\n",
                                     state[index].nesting_level,i);
                  i += length;
               } else if ((result = search_match(&string[i], mode->ignore, &length)) != NULL) {
                  count_code = 1;
                  if (debug) printf("Start ignore in column %d\n",i);
                  index++;
                  state[index].type = IN_IGNORE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else if ((result = search_match(&string[i], mode->comment, &length)) != NULL) {
                  item->comments += 1;
                  if (debug) printf("Start comment line %d in column %d\n",item->comments,i);
                  i += length;
                  index++;
                  state[index].type = IN_COMMENT;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
               } else if ((result = search_match(&string[i], mode->escape, &length)) != NULL) {
                  count_code = 1;
                  if (debug) printf("Start escape in column %d\n",i);
                  index++;
                  state[index].type = IN_ESCAPE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  i += length;
               } else if ((result = search_match(&string[i], mode->line, &length)) != NULL) {
                  if (debug) printf("Start line %d in column %d\n",
                                     item->code+1, i);
                  at_line_start = 1;
                  index++;
                  state[index].type = IN_LINE;
                  state[index].nesting_level = 0;
                  if (result->nestable) {
                     state[index].nest_marker = &result->start;
                  } else {
                     state[index].nest_marker = NULL;
                  }
                  state[index].end_marker = &result->end;
                  /*
                  ** For lines, we don't advance the line offset so we can
                  ** check for escapes, ignores, and alternative modes.
                  */
               } else {
                  /* No match, advance to the next character */
                  i += 1;
               }
               break;
            }
            /* No longer at the beginning of a line */
            if (i != start) at_start = 0;
         } /* end of physical line processing */
      } /* end of file processing */
      if (debug) printf("Code: %d Comments: %d\n",item->code,item->comments);
   }
   while (index > 0) {
      switch (state[index].type) {
      case START_STATE:
         /* Should happen, so we will just ignore it */
         break;
      case IN_COMMENT:
         fprintf(stderr,"Reached end of file while in a comment region\n");
         break;
      case IN_IGNORE:
         fprintf(stderr,"Reached end of file while in an ignore region\n");
         break;
      case IN_ESCAPE:
         fprintf(stderr,"Reached end of file while in an escape region\n");
         break;
      case IN_LINE:
         if (count_code) item->code += 1;
         count_code = 0;
         break;
      case ALTER_STATE:
         fprintf(stderr,"Reached end of file while in an alternative counting method region\n");
         break;
      }
      index--;
   }
   if (index != 0) {
      /* Unbalanced comments or ignore delimiters */
      fprintf(stderr,"Unexpected end of file.\n");
   }
   total_lines += item->lines;
   total_comments += item->comments;
   total_code += item->code;
   if (detailed) {
     if (brief) {
       /* Print a brief detailed report */
       fprintf(output,"%s\t%d\t%d\t%d\n",
	       item->name,item->lines,item->comments,item->code);
     } else {
       /* Print a full detailed report */
       fprintf(output,"%s ::= %d physical lines, %d lines of comments, and %d source lines of code\n",
	       item->name,item->lines,item->comments,item->code);
     }
   }

   fclose(fp);
}
